#!/usr/bin/env python3
"""
Zipity CLI Tool
A command-line interface for the Zipity file sharing service.
"""

import argparse
import requests
import sys
import os
from pathlib import Path
import json
from datetime import datetime

class ZipityCLI:
    def __init__(self, base_url="http://localhost:5000"):
        self.base_url = base_url.rstrip('/')
        self.upload_url = f"{self.base_url}/api/upload"
        self.status_url = f"{self.base_url}/api/status"
    
    def upload_file(self, file_path):
        """Upload a file and return the share URL"""
        if not os.path.exists(file_path):
            print(f"Error: File '{file_path}' not found")
            return False
        
        file_size = os.path.getsize(file_path)
        if file_size > 100 * 1024 * 1024:  # 100MB limit
            print("Error: File size exceeds 100MB limit")
            return False
        
        try:
            with open(file_path, 'rb') as f:
                files = {'file': (os.path.basename(file_path), f, 'application/octet-stream')}
                response = requests.post(self.upload_url, files=files, timeout=30)
            
            if response.status_code == 200:
                data = response.json()
                if data.get('success'):
                    print(f"✅ File uploaded successfully!")
                    print(f"📁 File: {data['original_filename']}")
                    print(f"📏 Size: {self._format_size(data['file_size'])}")
                    print(f"🔗 Share URL: {data['share_url']}")
                    print(f"⏰ Expires: {data['expires_at']}")
                    print(f"🆔 File ID: {data['file_id']}")
                    return True
                else:
                    print(f"❌ Upload failed: {data.get('error', 'Unknown error')}")
                    return False
            else:
                print(f"❌ Upload failed with status code: {response.status_code}")
                if response.text:
                    try:
                        error_data = response.json()
                        print(f"Error: {error_data.get('error', 'Unknown error')}")
                    except:
                        print(f"Error: {response.text}")
                return False
                
        except requests.exceptions.ConnectionError:
            print(f"❌ Error: Could not connect to {self.base_url}")
            print("Make sure the Zipity server is running")
            return False
        except requests.exceptions.Timeout:
            print("❌ Error: Upload timed out")
            return False
        except Exception as e:
            print(f"❌ Error: {e}")
            return False
    
    def check_status(self, file_id):
        """Check the status of an uploaded file"""
        try:
            response = requests.get(f"{self.status_url}/{file_id}")
            
            if response.status_code == 200:
                data = response.json()
                print(f"📁 File: {data['original_filename']}")
                print(f"📏 Size: {self._format_size(data['file_size'])}")
                print(f"📅 Uploaded: {data['upload_time']}")
                print(f"🔍 Accessed: {'Yes' if data['is_accessed'] else 'No'}")
                if data['access_time']:
                    print(f"📥 Downloaded: {data['access_time']}")
                print(f"⏰ Expires: {data['expires_at']}")
                print(f"💀 Expired: {'Yes' if data['is_expired'] else 'No'}")
                return True
            elif response.status_code == 404:
                print(f"❌ File with ID '{file_id}' not found")
                return False
            else:
                print(f"❌ Error checking status: {response.status_code}")
                return False
                
        except requests.exceptions.ConnectionError:
            print(f"❌ Error: Could not connect to {self.base_url}")
            return False
        except Exception as e:
            print(f"❌ Error: {e}")
            return False
    
    def _format_size(self, size_bytes):
        """Format file size in human readable format"""
        if size_bytes == 0:
            return "0 B"
        size_names = ["B", "KB", "MB", "GB"]
        i = 0
        while size_bytes >= 1024 and i < len(size_names) - 1:
            size_bytes /= 1024.0
            i += 1
        return f"{size_bytes:.1f} {size_names[i]}"

def main():
    parser = argparse.ArgumentParser(
        description="Zipity CLI - Secure file sharing from the command line",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  zipity upload document.pdf
  zipity upload /path/to/large-file.zip
  zipity status abc123-def456-ghi789
  zipity --server https://zipity.example.com upload file.txt
        """
    )
    
    parser.add_argument(
        '--server', '-s',
        default='http://localhost:5000',
        help='Zipity server URL (default: http://localhost:5000)'
    )
    
    subparsers = parser.add_subparsers(dest='command', help='Available commands')
    
    # Upload command
    upload_parser = subparsers.add_parser('upload', help='Upload a file')
    upload_parser.add_argument('file', help='Path to file to upload')
    
    # Status command
    status_parser = subparsers.add_parser('status', help='Check file status')
    status_parser.add_argument('file_id', help='File ID to check')
    
    args = parser.parse_args()
    
    if not args.command:
        parser.print_help()
        return
    
    cli = ZipityCLI(args.server)
    
    if args.command == 'upload':
        success = cli.upload_file(args.file)
        sys.exit(0 if success else 1)
    elif args.command == 'status':
        success = cli.check_status(args.file_id)
        sys.exit(0 if success else 1)

if __name__ == '__main__':
    main()
