# Zipity CLI Client

A command-line interface for the Zipity secure file sharing service.

## Installation

### Using pip (Recommended)

```bash
# Install from PyPI (when available)
pip install zipity-cli

# Or install from source
pip install git+https://github.com/your-username/zipity-cli.git
```

### Manual Installation

1. **Create a virtual environment:**
   ```bash
   python -m venv zipity-env
   source zipity-env/bin/activate  # On Windows: zipity-env\Scripts\activate
   ```

2. **Install dependencies:**
   ```bash
   pip install -r requirements.txt
   ```

3. **Make the script executable:**
   ```bash
   chmod +x zipity_cli.py
   ```

4. **Add to PATH (optional):**
   ```bash
   # Create a symlink
   sudo ln -s $(pwd)/zipity_cli.py /usr/local/bin/zipity
   
   # Or add to your shell profile
   echo 'export PATH="$PATH:$(pwd)"' >> ~/.bashrc
   source ~/.bashrc
   ```

## Usage

### Upload a file
```bash
python zipity_cli.py upload document.pdf
```

### Check file status
```bash
python zipity_cli.py status abc123-def456-ghi789
```

### Use custom server
```bash
python zipity_cli.py --server https://zipity.example.com upload file.txt
```

### Get help
```bash
python zipity_cli.py --help
```

## Configuration

### Environment Variables

- `ZIPITY_SERVER`: Default server URL (default: http://localhost:5000)
- `ZIPITY_API_KEY`: API key for authentication (if implemented)

### Configuration File

Create `~/.zipity/config`:
```ini
[default]
server = https://zipity.example.com
api_key = your-api-key-here
```

## Examples

### Basic file upload
```bash
$ python zipity_cli.py upload presentation.pptx
✅ File uploaded successfully!
📁 File: presentation.pptx
📏 Size: 2.5 MB
🔗 Share URL: https://zipity.example.com/download/abc123
⏰ Expires: 2024-01-02T10:30:00
🆔 File ID: abc123-def456-ghi789
```

### Check upload status
```bash
$ python zipity_cli.py status abc123-def456-ghi789
📁 File: presentation.pptx
📏 Size: 2.5 MB
📅 Uploaded: 2024-01-01T10:30:00
🔍 Accessed: No
⏰ Expires: 2024-01-02T10:30:00
💀 Expired: No
```

### Batch upload
```bash
# Upload multiple files
for file in *.pdf; do
    python zipity_cli.py upload "$file"
done
```

### Integration with other tools
```bash
# Upload and get URL for sharing
URL=$(python zipity_cli.py upload file.txt | grep "Share URL" | cut -d' ' -f3)
echo "Share this file: $URL"
```

## API Reference

The CLI client uses the following API endpoints:

- `POST /api/upload` - Upload a file
- `GET /api/status/{file_id}` - Check file status
- `GET /download/{token}` - Download a file

## Troubleshooting

### Common Issues

1. **Connection refused**: Make sure the Zipity server is running
2. **File too large**: Check server's MAX_CONTENT_LENGTH setting
3. **Permission denied**: Ensure the script is executable

### Debug Mode

Enable verbose output:
```bash
python zipity_cli.py --verbose upload file.txt
```

### Logs

Check server logs for detailed error information:
```bash
# Docker
docker-compose logs -f zipity

# Local
tail -f app.log
```

## License

This CLI client is part of the Zipity project and is available under the same license.

## Support

For issues and questions:
- Check the main Zipity documentation
- Open an issue on the project repository
- Contact the development team
